/*
 * Gestionnaire de Tâches CLI - Node.js avec readline-sync
 * Projet d'exercice sur les tableaux JavaScript
 */

import { ListeDeTaches } from './ListeDeTaches.js';
import readline from 'readline-sync';

const FICHIER_SAUVEGARDE = "taches.json";

const listeDeTaches = new ListeDeTaches(FICHIER_SAUVEGARDE);

/**
 * Affiche le titre de l'application
 */
function afficherTitre() {
    console.log('╔══════════════════════════════════╗');
    console.log('║    GESTIONNAIRE DE TÂCHES CLI    ║');
    console.log('╚══════════════════════════════════╝');
}

/**
 * Affiche le menu principal
 */
function afficherMenu() {
    console.log('\=== MENU PRINCIPAL ===');
    console.log('1. Afficher toutes les tâches');
    console.log('2. Afficher les statistiques');
    console.log('3. Marquer une tâche comme terminée');
    console.log('4. Ajouter une tâche');
    console.log('5. Supprimer une tâche');
    console.log('6. Supprimer toutesles tâches terminées');
    console.log('7. Trier les tâches');
    console.log('8. Recherche de tâches par mot clé');
    console.log('9. Sauvegarder et quitter');
    console.log('0. Quitter sans sauvegarder');
    console.log('=====================\n');
}


/**
 * Fonction principale de l'application
 */
function main() {

    let continuer = true;

    while (continuer) {
        afficherTitre();
        // afficherStatistiques();
        afficherMenu();

        const choix = readline.question('Votre choix : ');

        switch (choix) {
            case '1':
                afficher();
                break;
            case '2':
                afficherStatistiques();
                break;
            case '3':
                marquerTerminee();
                break;
            case '4':
                ajouterTache();
                break;
            case '5':
                supprimerTache();
                break;
            case '6':
                supprimerTachesTerminees();
                break;
            case '7':
                trierTaches();
                break;
            case '8':
                console.log("opération non encore supportée");
                break;
            case '9':
                if (sauvegarderTaches()) {
                    continuer = false;
                }
                break;
            case '0':
                const confirmation = readline.keyInYNStrict('Quitter sans sauvegarder ?');
                if (confirmation) {
                    continuer = false;
                }
                break;
            default:
                console.log('\n❌ Choix invalide!');
        }

        if (continuer) {
            readline.question('\nAppuyez sur Entree pour continuer...');
        }
    }

    console.log('\n👋 Au revoir!\n');
}

/**
 * Ajoute une nouvelle tâche à la liste
 */
function ajouterTache() {
    console.log('\n=== AJOUTER UNE TÂCHE ===');

    const description = readline.question('Description de la tache : ');

    if (description.trim() === '') {
        console.log('❌ La description ne peut pas être vide!');
        return;
    }

    let priorite = readline.keyInSelect(
        ['Basse', 'Moyenne', 'Haute'],
        'Choisissez la priorite :',
        { cancel: false }
    );

    priorite = ['Basse', 'Moyenne', 'Haute'][priorite];

    listeDeTaches.nouvelleTache(description, priorite)
    console.log('✅ Tâche ajoutée avec succès!');
}

/**
 * Affiche toutes les tâches
 * @param {Array} tableauTaches - Le tableau de tâches à afficher (optionnel)
 */
function afficher() {
    console.log('\n=== LISTE DES TÂCHES ===');

    if (listeDeTaches.nbTaches === 0) {
        console.log('Aucune tâche à afficher.');
        return;
    }
    listeDeTaches.afficherLesTaches();
}

/**
 * Marque une tâche comme terminée
 */
function marquerTerminee() {
    if (listeDeTaches.nbTaches === 0) {
        console.log('\nAucune tâche disponible.');
        return;
    }

    console.log('\n=== MARQUER COMME TERMINÉE ===' );
    listeDeTaches.afficherLesTaches();

    const choix = readline.questionInt('\nNumero de la tâche à marquer (0 pour annuler) : ');

    if (choix === 0) return;

    if (choix < 1 || choix > listeDeTaches.nbTaches) {
        console.log('❌ Numéro invalide!');
        return;
    }

    listeDeTaches.terminer(choix);
    console.log('✅ Statut modifié avec succès!');
}

/**
 * Supprime une tâche du tableau
 */
function supprimerTache() {
    if (listeDeTaches.nbTaches === 0) {
        console.log('\nAucune tâche à supprimer.');
        return;
    }

    console.log('\n=== SUPPRIMER UNE TÂCHE ===');
    listeDeTaches.afficherLesTaches();

    const choix = readline.questionInt('\nNumero de la tache à supprimer (0 pour annuler) : ');

    if (choix === 0) return;

    if (choix < 1 || choix > listeDeTaches.nbTaches) {
        console.log('❌ Numéro invalide!');
        return;
    }

    const confirmation = readline.keyInYNStrict('Confirmer la suppression ?');

    if (confirmation) {
        let tacheSupprimee = listeDeTaches.supprimer(choix);
        console.log(`✅ Tâche "${tacheSupprimee.description}" supprimée!`);
    } else {
        console.log('Suppression annulée.');
    }
}

/**
 * Affiche les statistiques des tâches
 */
function afficherStatistiques() {
    console.log('\n=== STATISTIQUES ===');

    const total = listeDeTaches.nbTaches;
    const terminees = listeDeTaches.nbTachesTerminees;
    const enCours = total - terminees;
    const prioriteHaute = listeDeTaches.nbTachesDePriorite('Haute');
    const prioriteMoyenne = listeDeTaches.nbTachesDePriorite('Moyenne');;
    const prioriteBasse = listeDeTaches.nbTachesDePriorite('Basse');

    console.log(`\nNombre total de tâches : ${total}`);
    console.log(`✓ Terminées : ${terminees}`);
    console.log(`○ En cours : ${enCours}`);

    if (total > 0) {
        const pourcentage = ((terminees / total) * 100).toFixed(1);
        console.log(`\nProgression : ${pourcentage}%`);

        console.log('\nRépartition par priorité :');
        console.log(`   Haute : ${prioriteHaute}`);
        console.log(`   Moyenne : ${prioriteMoyenne}`);
        console.log(`   Basse : ${prioriteBasse}`);
    }
}

/**
 * Supprime toutes les tâches terminées
 */
function supprimerTachesTerminees() {
    const terminees = listeDeTaches.nbTachesTerminees;

    if (terminees === 0) {
        console.log('\nAucune tâche terminée à supprimer.');
        return;
    }

    console.log(`\n${terminees} tâche(s) terminée(s) trouvée(s).`);
    const confirmation = readline.keyInYNStrict('Confirmer la suppression de toutes les taches terminees ?');

    if (confirmation) {
        listeDeTaches.supprimerTachesTerminees();
        console.log(`✅ ${terminees} tâche(s) supprimée(s)!`);
    } else {
        console.log('Suppression annulée.');
    }
}

/**
 * Trie les tâches selon différents critères
 */
function trierTaches() {
    if (listeDeTaches.nbTaches === 0) {
        console.log('\nAucune tâche à trier.');
        return;
    }

    console.log('\n=== TRIER LES TÂCHES ===');

    const critere = readline.keyInSelect(
        ['Par date (plus récentes)', 'Par date (plus anciennes)', 'Par priorité', 'Par statut'],
        'Choisissez le critere de tri :'
    );

    // critere est la valeur de l'index du critère de tri choisi
    // 0 : tri par date (les plus récentes)
    // 1 : tri par date (les plus anciennes)
    // 2 : tri par priorité (de la plus haute à la plus basse)
    // 3 : tri par statut (terminée ou non terminée)
    if (critere === -1) return; // Annuler

    switch (critere) {
        case 0: // Plus récentes
            listeDeTaches.trier((a, b) => b.id - a.id);
            break;
        case 1: // Plus anciennes
            listeDeTaches.trier((a, b) => a.id - b.id);
            break;
        case 2: // Par priorité
            const ordrePriorite = { 'Haute': 3, 'Moyenne': 2, 'Basse': 1 };
            listeDeTaches.trier((a, b) => ordrePriorite[b.priorite] - ordrePriorite[a.priorite]);
            break;
        case 3: // Par statut
            listeDeTaches.trier((a, b) => a.terminee - b.terminee);
            break;
    }

    console.log('✅ Tâches triées!');
    afficher();
}

/**
 * Sauvegarde les tâches dans un fichier JSON
 */
function sauvegarderTaches() {
    try {
        listeDeTaches.sauvegarderTaches(FICHIER_SAUVEGARDE);
        console.log('\n💾 Tâches sauvegardées avec succès!');
        return true;
    } catch (error) {
        console.log('\n❌ Erreur lors de la sauvegarde : ' + error.message);
        return false;
    }
}

// Lancement de l'application
main();
