/*
 * © Philippe GENOUD - Université Grenobles Alpes
 * Equipe STeamer - Laboratoire d'Informatique de Grenoble
 *
 */

/**
 * VisageExpressif.js : Classe permettant de définir le type objet correspondant à des
 * visages 'rebondissants' qui changent d'expression chaque fois qu'ils touchent l'un des
 * bords du canvas où ils se déplacent.
 */
import {Visage} from './Visage.js';

export class VisageExpressif extends Visage {
	/**
	 * les couleurs associées aux différentes expression
	 *  @type Array
	 */
	static couleurs = [
		"#b7daff", // couleur visage impassible
		"yellow", // couleur visage joyeux
		"#4caf50", // couleur visage triste
	];

	/**
	 * Constructeur pour des objets VisageExpressif.
	 * Par defaut à la création le VisageExpressifsera avec l'expression impassible
	 * @param {Canvas} canvas le canvas dans lequel le visage se déplace
	 * @param {number} xCentre abscisse du centre du visage
	 * @param {number} yCentre ordonnée du centre du visage
	 * @param {number} [rayon=50] rayon du visage
	 * @param {number} [vx = 5] déplacement horizontal élémentaire du visage
	 * @param {number} [vy = 5] déplacement vertical élémentaire du visage
	 * @returns {Visage}
	 */
	constructor(canvas, xCentre, yCentre, rayon = 50, vx = 5, vy = 5) {
		super(canvas, xCentre, yCentre, rayon, vx, vy); // appel constructeur de la super classe (Visage)
		// pour définir les propriétés correspondant à un Visage

		// propriétés supplémentaires pour les Visages expressifs
		this.expression = 0;
		this.couleur = VisageExpressif.couleurs[this.expression];
	}

	/**
	 * redéfinition de la méthode héritée pour prendre en compte l'expression du VisageExpressif
	 * @returns {undefined}
	 */
	_dessinerBouche() {
		switch (this.expression) {
			case 0:
				this._dessinerBoucheImpassible();
				break;
			case 1: // visage joyeux
				super._dessinerBouche();
				break;
			default: // visage triste
				this._dessinerBoucheTriste();
				break;
		}
		// équivalent à
		// if (this.expression === 1) { // visage joyeux
		// 	super._dessinerBouche();
		// } else if (this.expression == 0) {
		// 	this._dessinerBoucheImpassible();
		// } else {
		// 	this._dessinerBoucheTriste();
		// }
	}

	_dessinerBoucheImpassible() {
		this.ctx.beginPath(); // beginPath permet de définir un nouveau chemin de tracé, en
		// l'absence de cette instruction on aurait un trait reliant le point de début de
		// tracé du cercle délimitant le visage au point de début du cercle
		// matérialisant la bouche
		this.ctx.arc(this.xCentre, this.yCentre - 1.3 * this.rayon, this.rayon * 1.8, 0.4 * Math.PI, 0.6 * Math.PI, false);
		this.ctx.strokeStyle = "red";
		this.ctx.stroke();
	}

	_dessinerBoucheTriste() {
		this.ctx.beginPath(); // beginPath permet de définir un nouveau chemin de tracé, en
		// l'absence de cette instruction on aurait un trait reliant le point de début de
		// tracé du cercle délimitant le visage au point de début du cercle
		// matérialisant la bouche
		this.ctx.arc(this.xCentre, this.yCentre + 0.8 * this.rayon, this.rayon * 0.5, (6 / 5) * Math.PI, (9 / 5) * Math.PI, false);
		this.ctx.strokeStyle = "red";
		this.ctx.stroke();
	}

	/**
	 * redéfinition de la méthode deplacer() héritée de Visage pour prendre en compte le
	 * changement d'expression et couleur lorsqu'il y a rebond.
	 * @returns {undefined}
	 */
	deplacer() {
		if (super.deplacer()) {
			// un bord du canvas a été touché
			this.expression = (this.expression + 1) % 3;
			this.couleur = VisageExpressif.couleurs[this.expression];
		}
	}
}
