import { Joueur } from "./Joueur.js";
import { Jeu32Cartes } from "./Jeu32Cartes.js";

/**
 * Classe Bataille
 * Gère une partie du jeu de bataille entre deux joueurs
 * voir  https://fr.wikipedia.org/wiki/Bataille_(jeu)
 */
export class Bataille {
    /**
     * Constructeur
     * @param {string} nom1 - Le nom du joueur 1
     * @param {string} nom2 - Le nom du joueur 2
     */
    constructor(nom1, nom2) {
        this.joueur1 = new Joueur(nom1);
        this.joueur2 = new Joueur(nom2);
        this.leJeuDeCartes = new Jeu32Cartes();
    }

    /**
     * Recupère la référence de l'un des joeurs de la partie de bataille
     * @param {number} numeroJoueur le numéro du joeur à récupérer
     * @returns {Joueur} Le joueur de numéro numeroJoueur. Renvoie null
     * si il n'y a  pas de joueur correspondant au numéro 
     */
    getJoueur(numeroJoueur) {
        if (numeroJoueur === 1) return this.joueur1;
        else if (numeroJoueur === 2) return this.joueur2;
        else return null;
    }

    /**
     * Distribue les cartes aux deux joueurs
     * @param {number} nb nombre de cartes à donner à chaque joueur, par défaut 16
     */
    distribuerCartes(nb = 16) {
        this.leJeuDeCartes.melanger();
        for (let i = 0; i < nb; i++) {
            this.joueur2.ajouterCarteSurSaPile(this.leJeuDeCartes.retirerCarte());
            this.joueur1.ajouterCarteSurSaPile(this.leJeuDeCartes.retirerCarte());
        }
    }

    /**
     * Joue un tour : chaque joueur joue une carte
     */
    jouerUnTour() {
        this.joueur1.jouerCarte();
        this.joueur2.jouerCarte();
    }

    /**
     * Donne toutes les cartes du pli au joueur gagnant qui les mets 
     * dans son tas.
     * @param {Joueur} joueur - Le joueur qui remporte le pli
     */
    donnerLesCartesDuPliA(joueur) {
        if (joueur === this.joueur1) {
            this.joueur1.prendreCartesJoueesPar(this.joueur1);
            this.joueur1.prendreCartesJoueesPar(this.joueur2);
        } else {
            this.joueur2.prendreCartesJoueesPar(this.joueur2);
            this.joueur2.prendreCartesJoueesPar(this.joueur1);
        }
    }

    /**
     * Détermine le gagnant du tour
     * @returns {Joueur|null} Le joueur gagnant ou null en cas d'égalité
     */
    gagnantDuTour() {
        if (this.joueur1.carteJouee.valeur.ordinal > this.joueur2.carteJouee.valeur.ordinal) {
            // Le joueur 1 a gagné le tour
            return this.joueur1;
        } else if (this.joueur2.carteJouee.valeur.ordinal > this.joueur1.carteJouee.valeur.ordinal) {
            // Le joueur 2 a gagné le tour
            return this.joueur2;
        } else {
            // Les deux cartes ont la même valeur
            // Mais le tour s'arrête si un joueur n'a plus de cartes
            if (this.joueur1.nbCartes > 0 && this.joueur2.nbCartes === 0) {
                return this.joueur1;
            } else if (this.joueur2.nbCartes > 0 && this.joueur1.nbCartes === 0) {
                return this.joueur2;
            } else {
                // pas de gagnant
                return null;
            }
        }
    }

    /**
     * Détermine le gagnant de la partie, c'est à dire le joueur qui a le plus de cartes dans son tas.
     *
     * @returns {Joueur|null} Le joueur gagnant ou null en cas d'égalité
     */
    gagnantPartie() {
        if (this.joueur1.nbCartes > this.joueur2.nbCartes) {
            return this.joueur1;
        } else if (this.joueur1.nbCartes < this.joueur2.nbCartes) {
            return this.joueur2;
        } else {
            // Les deux joueurs ont le même nombre de cartes, pas de gagnant
            return null;
        }
    }

    /**
     * @returns {boolean} true si la partie est finie
     */
    partieFinie() {
        return this.joueur1.nbCartes === 0 || this.joueur2.nbCartes === 0;
    }

    /**
     * ramasse toutes les cartes des joueurs (celles qu'il a en main plus éventuellement
     * celles qui sont dans sa pile de cartes jouées) et les remet dans le jeu de 32 cartes
     * qui est mélangé.
     * 
     * Après cette opération, les joueurs n'ont plus aucune carte (leurs piles de cartes sont vides).
     */
    terminerPartie() {
        this.leJeuDeCartes.reprendreCartes(this.joueur1.rendreToutesLesCartes());
        this.leJeuDeCartes.reprendreCartes(this.joueur2.rendreToutesLesCartes());
        this.leJeuDeCartes.melanger();
    }
}
