/**
 * BatailleGUI.js
 * Contrôleur pour l'interface graphique du jeu de Bataille
 */
import { Bataille } from "./Bataille.js";

const PATH_TO_IMAGES = "./images/";

class BatailleGUI {
    constructor(nomJoueur1, nomJoueur2) {
        this.bataille = new Bataille(nomJoueur1, nomJoueur2);

        document.getElementById("nomJoueur1").textContent = nomJoueur1;
        document.getElementById("nomJoueur2").textContent = nomJoueur2;

        // Références aux éléments DOM qui seront éventuellement modifiés lors des interactions de l'utilisateur
        this.commentaire = document.getElementById("commentaire");
        this.btnJouer = document.getElementById("btnJouer");
        this.btnRemporter = document.getElementById("btnRemporter");
        this.btnTerminerPartie = document.getElementById("btnTerminerPartie");
        this.btnNouvellePartie = document.getElementById("btnNouvellePartie");
        this.btnDistribuer = document.getElementById("btnDistribuer");

        this.divDemarrerPartie = document.getElementById("demarrerPartie");
        this.divPartieEnCours = document.getElementById("partieEnCours");

        // Gestionnaires d'événements
        this.btnDistribuer.addEventListener("click", () => this.debuterPartie());
        this.btnJouer.addEventListener("click", () => this.jouerUneCarte());
        this.btnRemporter.addEventListener("click", () => this.remporter());
        this.btnNouvellePartie.addEventListener("click", () => this.nouvellePartie());
        this.btnTerminerPartie.addEventListener("click", () => this.afficherFinPartie());
    }

    /**
     * Gère le clic sur le bouton "Distribuer"
     */
    debuterPartie() {
        // distribuer les cartes
        let nbCartes = parseInt(document.getElementById("nbCartes").value);
        this.bataille.distribuerCartes(nbCartes);

        // Afficher les données des joueurs
        this.afficherEtatDuJoueur(1);
        this.afficherEtatDuJoueur(2);

        //  masquer le div contenant l'input nbCartes et le bouton distribuer
        this.divDemarrerPartie.classList.add("hidden");

        // rendre visible le div contenant les boutons controlant le déroulement
        // de la partie en cours
        this.divPartieEnCours.classList.remove("hidden");

        this.btnTerminerPartie.disabled = false;
        this.btnJouer.disabled = false;
        this.commentaire.textContent = "Les cartes ont été distribuées, cliquez sur Jouer pour commencer ou sur Terminer pour interrompre la partie.";
    }

    /**
     * Gère le clic sur le bouton "Jouer"
     */
    jouerUneCarte() {
        this.bataille.jouerUnTour();

        // Afficher les données des joueurs
        this.afficherEtatDuJoueur(1);
        this.afficherEtatDuJoueur(2);

        // Vérifier s'il y a un gagnant du tour
        const joueurGagnant = this.bataille.gagnantDuTour();
        if (joueurGagnant !== null) {
            // un des deux joueurs a gagné le tour
            this.commentaire.textContent = joueurGagnant.nom + " gagne ce tour. Cliquez sur Remporter pour qu'il(elle) ramasse les cartes jouées !";
            // désactiver le bouton Jouer
            this.btnJouer.disabled = true;
            // activer le bouton Remporter
            this.btnRemporter.disabled = false;
        } else {
            if (this.bataille.partieFinie()) {
                // il y  bataille mais un des deux joueurs n'a plus de cartes
                this.afficherFinPartie();
            } else {
                this.commentaire.textContent = "Bataille ! On rejoue...";
            }
        }
    }

    /**
     * Gère le clic sur le bouton "Remporter"
     */
    remporter() {
        // désactiver le bouton Remporter
        this.btnRemporter.disabled = true;

        const joueurGagnant = this.bataille.gagnantDuTour();

        if (joueurGagnant === null) {
            // Vérifier si la partie est finie (cas où les deux joueurs
            // n'ont plus de carte)
            if (this.bataille.partieFinie()) {
                this.afficherFinPartie();
            } else {
                this.commentaire.textContent = "Bataille ! on rejoue...";
            }
        } else {
            // le joueur gagnant ramasse les cartes qui ont été jouées
            this.bataille.donnerLesCartesDuPliA(joueurGagnant);

            // afficher les données des joueurs
            this.afficherEtatDuJoueur(1, null);
            this.afficherEtatDuJoueur(2, null);

            if (this.bataille.partieFinie()) {
                this.afficherFinPartie();
            } else {
                this.commentaire.textContent = joueurGagnant.nom + " a ramassé le pli, vous pouvez rejouer.";
                // Réactiver le bouton Jouer
                this.btnJouer.disabled = false;
            }
        }
    }

    /**
     * Gère le clic sur le bouton "Nouvelle Partie"
     */
    nouvellePartie() {

        // TODO
        //  à vous de compléter

        // ramasse toutes les cartes des joueurs et les mets dans le jeu de 32 cartes
        // qui est mélangé et prêt pour une nouvelle distribution de cartes.
        

        // Réinitialiser l'interface
        // rendre visible le div contenant l'input nbCartes et le bouton distribuer

        // masquer le div contenant les boutons controlant le déroulement
        // de la partie en cours

        //  remmettre les boutons dans le bon état disable ou non
  
        // afficher données des joueurs

        // afficher dans la zone commentaires le message 
        // "Nouvelle partie ! Cliquez sur 'Distribuer' pour commencer."

 
    }

    /**
     * Modifie l'interface pour afficher la fin d'une partie
     */
    afficherFinPartie() {
        const gagnant = this.bataille.gagnantPartie();
        let message = "Partie terminée ! ";

        if (gagnant) {
            message += gagnant.nom + " a gagné !";
        } else {
            message += "Match nul ! Les joueurs ont le même nombre de cartes";
        }
        this.commentaire.textContent = message;

        // désactiver les boutons Jouer et Remporter
        this.btnJouer.disabled = true;
        this.btnRemporter.disabled = true;
        // masquer le bouton terminer partie
        this.btnTerminerPartie.classList.add("hidden");
        // faire apparaître à la place le bouton Nouvelle Partie
        this.btnNouvellePartie.classList.remove("hidden");
    }

     /**
     * Affiche les données du joueur
     * - Son tas de carte et le nombre de cartes qu'il contient
     * - le tas de cartes qu'il joué (la dernière carte jouée est retournée)
     *   et le nombre de carte qu'il contient
     * @param {number} numeroJoueur
     */
    afficherEtatDuJoueur(numeroJoueur) {
        const joueur = this.bataille.getJoueur(numeroJoueur);
        const carte = joueur.carteJouee;

        const carteJoueurImg = document.getElementById("carteJoueur" + numeroJoueur);
        const tasJoueurImg = document.getElementById("tasJoueur" + numeroJoueur);
        const nbCartesSpan = document.getElementById("nbCartes" + numeroJoueur);
        const nbCartesJoueesSpan = document.getElementById("nbCartesJouees" + numeroJoueur);

        if (carte) {
            carteJoueurImg.src = PATH_TO_IMAGES + carte.nomFichierImage;
            carteJoueurImg.alt = carte.nom;
            carteJoueurImg.classList.remove("card-empty");
        } else {
            carteJoueurImg.classList.add("card-empty");
        }

        const nbCartes = joueur.nbCartes;
        if (nbCartes === 0) {
            tasJoueurImg.classList.add("card-empty");
        } else {
            tasJoueurImg.classList.remove("card-empty");
        }

        nbCartesSpan.textContent = nbCartes;
        nbCartesJoueesSpan.textContent = joueur.nbCartesJouees;
    }
}

// Initialisation au chargement de la page
document.addEventListener("DOMContentLoaded", () => {
    const batailleGUI = new BatailleGUI("Anna Tomie", "Guy de Michelin");
});
