import { PileCartes } from './PileCartes.js';

/**
 * Classe Joueur: représente un joueur de bataille.
 * 
 * Un joueur a les attributs suivants :
 * 
 * - nom : son nom complet (prénom + nom)
 * - cartesEnMain : la pile des cartes qu'il possède pour jouer
 * - cartesJouees : la pile des cartes qu'il a jouées
 */
export class Joueur {
    /**
     * Constructeur
     * @param {string} nom - Le nom du joueur
     */
    constructor(nom) {
        this.nom = nom;
        this.cartesEnMain = new PileCartes();
        this.cartesJouees = new PileCartes();
    }

    /**
     * @returns {number} Le nombre de cartes présentes dans le tas du joueur
     */
    get nbCartes() {
        return this.cartesEnMain.nbCartes;
    }

    /**
     * @returns {number} Le nombre de cartes présentes dans la pile des cartes
     *       jouées par le joueur.
     */
    get nbCartesJouees() {
        return this.cartesJouees.nbCartes;
    }

    /**
     * Renvoie la dernière carte jouée par le joueur, c'est à dire la carte
     * qui est au sommet de sa pile des cartes jouées. Renvoie null si le joueur n'a
     * pas encore joué de carte (la pile des cartes jouées est vide).
     */
    get carteJouee() {
        return this.cartesJouees.sommet;
    }

    /**
     * Le joueur joue la carte qu'il a au sommet de son tas, il la retire de sa pile
     * cartesEnMain et la dépose sur sa pile de cartesJouee
     */
    jouerCarte() {
        this.cartesJouees.ajouterCarteSurLaPile(this.cartesEnMain.retirerCarte());
    }

    /**
     * Le joueur ajoute une carte sur sa pile
     * @param {Carte} carte - La carte à ajouter
     * @throws {Error} Si le joueur a déjà toutes les cartes
     */
    ajouterCarteSurSaPile(carte) {
        this.cartesEnMain.ajouterCarteSurLaPile(carte);
    }

    /**
     * Le joueur ajoute une carte sous sa pile
     * @param {Carte} carte - La carte à ajouter
     * @throws {Error} Si le joueur a déjà toutes les cartes
     */
    ajouterCarteSousSaPile(carte) {
        this.cartesEnMain.insererCarteSousLaPile(carte);
    }

    prendreCartesJoueesPar(autreJoueur) {
        this.cartesEnMain.recuperer(autreJoueur.cartesJouees);
    }

    /**
     * Le joueur se débarasse de toutes les cartes qu'il a en sa possession,
     * c'est à dire toutes les cartes qui son dans son tas et toutes les cartes
     * qui sont dans sa pile de cartes jouées.
     * 
     * Une fois cette action effectuée, le joueur n'a plus de cartes (ses 2 piles sont vides)
     * 
     * @returns {Array[Cartes]} un tableau de cartes qui contient toutes les cartes 
     * dont le joueur s'est débarassé
     */
    rendreToutesLesCartes() {
        return this.cartesEnMain.vider().concat(this.cartesJouees.vider());
        // avecl'opérateur spread
        // return [... this.carteEnMain.lesCartes, ... this.cartesJouees.lesCartes];
    }

    /**
     * Affiche la main du joueur (pour debug)
     */
    afficherMain(short) {
        console.log("main de " + this.nom);
        this.cartesEnMain.afficher(short);
    }
}
