/**
 * Énumération représentant la valeur des cartes.
 * Les valeurs sont classées dans l'ordre selon la hiérarchie du bridge
 * https://fr.wikipedia.org/wiki/Bataille_(jeu)
 */
// export const ValeurCarte = Object.freeze({
//     SEPT: { nom: "sept", nomCourt : "7", ordinal: 0 },
//     HUIT: { nom: "huit", nomCourt : "8", ordinal: 1 },
//     NEUF: { nom: "neuf", nomCourt : "9", ordinal: 2 },
//     DIX: { nom: "dix", nomCourt : "10", ordinal: 3 },
//     VALET: { nom: "valet", nomCourt : "V", ordinal: 4 },
//     DAME: { nom: "dame", nomCourt : "D", ordinal: 5 },
//     ROI: { nom: "roi", nomCourt : "R", ordinal: 6 },
//     AS: { nom: "as", nomCourt : "1", ordinal: 7 }
// });

export class ValeurCarte {
    constructor(nom, nomCourt, ordinal) {
        this.nom = nom;
        this.nomCourt = nomCourt;
        this.ordinal = ordinal;
        Object.freeze(this); // Rend l'instance immuable
    }
    
    toString() {
        return this.nom;
    }
    
    /**
     * Compare cette valeur avec une autre
     * @param {ValeurCarte} autre 
     * @returns {number} -1 si inférieur, 0 si égal, 1 si supérieur
     */
    compareTo(autre) {
        return this.ordinal - autre.ordinal;
    }
    
    static SEPT = new ValeurCarte("sept", "7", 0);
    static HUIT = new ValeurCarte("huit", "8", 1);
    static NEUF = new ValeurCarte("neuf", "9", 2);
    static DIX = new ValeurCarte("dix", "10", 3);
    static VALET = new ValeurCarte("valet", "V", 4);
    static DAME = new ValeurCarte("dame", "D", 5);
    static ROI = new ValeurCarte("roi", "R", 6);
    static AS = new ValeurCarte("as", "1", 7);
    
    static values() {
        return [
            this.SEPT, 
            this.HUIT, 
            this.NEUF, 
            this.DIX, 
            this.VALET, 
            this.DAME, 
            this.ROI, 
            this.AS
        ];
    }
    
    static fromNom(nom) {
        return this.values().find(v => v.nom === nom);
    }
    
    static fromOrdinal(ordinal) {
        return this.values().find(v => v.ordinal === ordinal);
    }
}

Object.freeze(ValeurCarte); // Empêche d'ajouter de nouvelles valeurs